package gov.va.med.domain.service.messaging.encode.hl7;

import java.util.Date;

import gov.va.med.domain.model.Patient;
import gov.va.med.domain.model.Prescription;
import gov.va.med.domain.model.RxRefillPayload;
import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.service.messaging.MessagingConstants;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.encode.hl7.structure.OMP_O09;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7HeaderParameters;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7v24Helper;
import gov.va.med.domain.service.messaging.environment.EndPoint;
import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.DataTypeException;
import ca.uhn.hl7v2.model.v24.segment.ORC;
import ca.uhn.hl7v2.model.v24.segment.PID;
import ca.uhn.hl7v2.model.v24.segment.RXE;
import ca.uhn.hl7v2.parser.Parser;
import ca.uhn.hl7v2.parser.PipeParser;

import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;


/**
 * RxRefillEncoder
 * Encodes RxRefill Request message, using the OMP_O09 message structure.
 *
 * @author Slava Uchitel
 * @version $Id: RxRefillEncoder.java,v 1.5 2005/08/10 15:38:28 ye.jin Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class RxRefillEncoder extends HL7MessageEncoder implements MessagingConstants {
	private static final String EVENT = "OMP";
	private static final String TRIGGER = "O09";
	private static final String STRUCTURE = EVENT + "_" + TRIGGER;

	private OMP_O09 createRefillRequestMessage(Patient patient, String[] rxNumbers,
	                                           HL7HeaderParameters params)
	    throws DataTypeException, HL7Exception {
		OMP_O09 omp_o09 = new OMP_O09();
		//MSH
		params.setFieldSeparator(DEFAULT_FIELD_SEPARATOR);
		params.setDelimiters(DEFAULT_DELIMITERS);
		params.setEvent(EVENT);
		params.setTrigger(TRIGGER);
		params.setMessageStructure(STRUCTURE);
		params.setAcceptAcknowledgmentType(EMPTY_STRING);
		params.setApplicationAcknowledgmentType(EMPTY_STRING);

		HL7v24Helper.fillMSHSegment(omp_o09.getMSH(), params);

		//PID
		PID pid = omp_o09.getPID();
		if(patient.getSocialSecurityNumber() != null
		        	&& patient.getSocialSecurityNumber().trim().length() > 0)

		    HL7v24Helper.addIdToPidPID(pid,
			                           patient.getSocialSecurityNumber(),
			                           HL7v24Helper.ID_SSN, EMPTY_STRING, EMPTY_STRING);
		if(patient.getIcn() != null && patient.getIcn().length() > 0)
			HL7v24Helper.addIdToPidPID(pid,
			                           patient.getIcn(),
			                           HL7v24Helper.ID_ICN, EMPTY_STRING, EMPTY_STRING);

		buildRxeAndOrcSegments(rxNumbers, omp_o09);

		return omp_o09;
	}

	private void buildRxeAndOrcSegments(String[] rxNumbers, OMP_O09 omp_o09) throws DataTypeException {
		String currentTime = dateToyyyyMMddhhmmss(new Date());
		for(int i = 0; i < rxNumbers.length; i++) {
			RXE rxe = omp_o09.getORCRXE(i).getRXE();
			rxe.getQuantityTiming().getQuantity().getQuantity().setValue("1");
			rxe.getQuantityTiming().getStartDateTime().setValue(currentTime);
			rxe.getGiveCode().getIdentifier().setValue("RF");
			rxe.getGiveCode().getNameOfCodingSystem().setValue("HL70119");
			rxe.getGiveAmountMinimum().setValue("1");
			rxe.getGiveUnits().getIdentifier().setValue("1 refill unit");
			rxe.getPrescriptionNumber().setValue(rxNumbers[i]);

			ORC orc = omp_o09.getORCRXE(i).getORC();
			orc.getPlacerOrderNumber().getEntityIdentifier().setValue(rxNumbers[i] + "-" + currentTime);
			orc.getOrderControl().setValue("RF");
		}
	}

	public Object encode(IPayload payload, EndPoint ep)
	    throws MessagingException {
		RxRefillPayload rxRefillPayload = (RxRefillPayload)payload;
		Patient patient = rxRefillPayload.getPatient();
		Prescription[] prescriptions = rxRefillPayload.getPrescriptionList();

		String[] rxNumbers = new String[prescriptions.length];
		for(int i = 0; i < prescriptions.length; i++)
			rxNumbers[i] = String.valueOf(prescriptions[i].getPrescriptionNum());
		try {
			HL7HeaderParameters params = buildHL7HeaderParameters(ep, rxRefillPayload);
			OMP_O09 omp_o09 =
			    createRefillRequestMessage(patient, rxNumbers, params);
			Parser parser = new PipeParser();
			//return parser.encode(qbp_q13);
			return parser.encode(omp_o09);
		}
		catch(DataTypeException e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
		catch(HL7Exception e) {
			String msg = "Failed while encoding a request message into HL7.";
			getLogger().error(msg + ". Error: " + e.getMessage());
			throw new MessagingException(msg);
		}
	}

	private HL7HeaderParameters buildHL7HeaderParameters(EndPoint ep, RxRefillPayload rxRefillPayload) {
		HL7HeaderParameters params = new HL7HeaderParameters();
		params.setSrcStationNumber(ep.getEncodingParameter(SENDING_FACILITY));
		params.setSrcDomain(ep.getEncodingParameter(SENDING_FACILITY_DOMAIN));
		params.setSrcApplication(ep.getEncodingParameter(SENDING_APPLICATION));
		params.setDstStationNumber(ep.getEncodingParameter(RECEIVING_FACILITY));
		params.setDstDomain(ep.getEncodingParameter(RECEIVING_FACILITY_DOMAIN));
		params.setDstApplication(ep.getEncodingParameter(RECEIVING_APPLICATION));
		params.setSystemType(ep.getEncodingParameter(SYSTEM_TYPE));
		params.setMessageControlId(rxRefillPayload.getRequestId().toString());
		return params;
	}
}
